/*
* Copyright  2013 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include "host_interface.h"
#include "buffer.h"
#include "hardware.h"
#include "API.h"
#include "MSP430_hardware.h"

extern "C" {

#include "msp430.h"
#include "inc/hw_memmap.h"
#include "usci_a_uartbaudrate.h"
#include "usci_a_uart.h"
#include "wdt_a.h"
#include "gpio.h"
#include "ucs.h"

}

byte * readp;
byte * writep;
indextype readlen, readsize, writelen, writesize;

static Buffer<indextype, RS232_BUFFSIZE> recbuf;
static Buffer<indextype, RS232_BUFFSIZE> sendbuf;

void Host_Interface::Init(void) {
	//P3.4,5 = USCI_A0 TXD/RXD
	GPIO_setAsPeripheralModuleFunctionInputPin(GPIO_PORT_P4, GPIO_PIN4 + GPIO_PIN5);

	if (STATUS_FAIL
			== USCI_A_UART_init(__MSP430_BASEADDRESS_USCI_A1__, USCI_A_UART_CLOCKSOURCE_SMCLK,
			25000000, 115200,
					USCI_A_UART_NO_PARITY, USCI_A_UART_LSB_FIRST, USCI_A_UART_ONE_STOP_BIT,
					USCI_A_UART_MODE, USCI_A_UART_LOW_FREQUENCY_BAUDRATE_GENERATION)) {
		return;
	}
	//Enable UART module for operation
	USCI_A_UART_enable(__MSP430_BASEADDRESS_USCI_A1__);

	//Enable Receive Interrupt
	USCI_A_UART_enableInterrupt(__MSP430_BASEADDRESS_USCI_A1__,
			USCI_A_UART_RECEIVE_INTERRUPT);

	//interrupts enabled
	__bis_SR_register(GIE);
}

unsigned short Host_Interface::Read(unsigned char* buf, unsigned short len) {
	indextype x = 0;

	if(Received())
		x = recbuf.Read(buf, len);

	if (x) {
		return x;
	} else
		return 0;
}

unsigned short Host_Interface::Write(unsigned char* buf, unsigned short len) {

	GPIO::setSerialTX(true);

	while(len--)
	{
		USCI_A_UART_transmitData(__MSP430_BASEADDRESS_USCI_A1__, *buf++);
	    //USCI TX buffer ready?
	    while (!USCI_A_UART_getInterruptStatus(__MSP430_BASEADDRESS_USCI_A1__, UCTXIFG)) ;
	}

	GPIO::setSerialTX(false);

	return 0;
}

unsigned short Host_Interface::Received() {
	return recbuf.used();
}

void Host_Interface::run() {
}

//******************************************************************************
//
//This is the USCI_A1 interrupt vector service routine.
//
//******************************************************************************
static byte x = 0;
#pragma vector=USCI_A1_VECTOR
__interrupt void USCI_A1_ISR(void) {
	int state = __even_in_range(UCA1IV, 4);
	switch (state) {
	// Vector 0 - no interrupt
	case 0:
		break;
	//Vector 2 - RXIFG
	case 2:
		//Receive the data
		x = USCI_A_UART_receiveData(__MSP430_BASEADDRESS_USCI_A1__);
		recbuf.Write(&x, 1); //store received byte in recbuf
		break;
	// Vector 4 - TXIFG
	case 4: {
		USCI_A_UART_clearInterruptFlag(__MSP430_BASEADDRESS_USCI_A1__,UCA1IFG);
		if (!(sendbuf.empty())) //any data in writebuffer to be sent ?
		{
			byte x;
			if (sendbuf.Read(&x, 1)) //fetch byte from buffer
					{
				USCI_A_UART_transmitData(__MSP430_BASEADDRESS_USCI_A1__, x);
			}
		} else //no more data to send
		{
			USCI_A_UART_disableInterrupt(__MSP430_BASEADDRESS_USCI_A1__,
					USCI_A_UART_TRANSMIT_INTERRUPT); //Disable the transmit interrupt, otherwize TXE would cause another interrupt (because the transmit data register is still empty)
		}
		break;
	}
	default:
		break;
	}
}

#pragma vector= USB_UBM_VECTOR
__interrupt void USB_UBM_ISR (void)
{
	while(1);
}
